<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use PhpMqtt\Client\Facades\Mqtt;
use App\Models\Device;
use App\Models\DeviceLatestData;
use App\Models\DeviceDataLog;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class MqttDeviceListener extends Command
{
    protected $signature = 'mqtt:listen-devices';
    protected $description = 'Listen to MQTT telemetry from washroom devices';

    public function handle()
    {
        $this->info('Starting MQTT device listener...');

        $connection = Mqtt::connection();
        $connection->connect();

        $connection->subscribe('washroom/+/telemetry', function (string $topic, string $message) {

            try {
                // Topic format: washroom/{device_uid}/telemetry
                $topicParts = explode('/', $topic);
                $deviceUid = $topicParts[1] ?? null;

                if (!$deviceUid) {
                    return;
                }

                $payload = json_decode($message, true);


            if (!isset($payload['m'], $payload['t'])) {
                Log::warning('MQTT missing required fields', $payload);
                return;
            }

            // Normalize MAC
            $mac = strtoupper(trim($payload['m']));

            // Find device
            $device = Device::where('mac_address', $mac)->first();

            if (!$device) {
                Log::warning('Unknown device MAC', ['mac' => $mac]);
                return;
            }

            // Parse timestamp
            try {
                $timestamp = Carbon::parse($payload['t']);
            } catch (\Exception $e) {
                Log::warning('Invalid timestamp', ['t' => $payload['t']]);
                return;
            }



                if (!is_array($payload)) {
                    Log::warning("Invalid JSON payload from {$topic}");
                    return;
                }

                $device = Device::where('device_uid', $deviceUid)->first();

                if (!$device) {
                    Log::warning("Unknown device UID: {$deviceUid}");
                    return;
                }

                $now = Carbon::now();

                // 🔹 Update latest snapshot
                DeviceLatestData::updateOrCreate(
                    ['device_id' => $device->id],
                    [
                        'payload' => $payload,
                        'received_at' => $now,
                         'timestamp' => $timestamp,
                        'ammonia'   => $payload['a'] ?? null,
                        'sulphur'   => $payload['s'] ?? null,
                        'voc'       => $payload['v'] ?? null,
                    ]
                );

                // 🔹 Insert history log
                DeviceDataLog::create([
                    'device_id' => $device->id,
                    'payload' => $payload,
                    'received_at' => $now,
                    'timestamp' => $timestamp,
                    'ammonia'   => $payload['a'] ?? null,
                    'sulphur'   => $payload['s'] ?? null,
                    'voc'       => $payload['v'] ?? null,
                ]);

                Log::info("MQTT data stored for device {$deviceUid}");

            } catch (\Throwable $e) {
                Log::error('MQTT processing error: ' . $e->getMessage());
            }

        }, 1);

        // 🔁 Start blocking loop
        $connection->loop(true);
    }
}
